/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.asic.common;

import eu.europa.esig.dss.asic.common.ASiCContent;
import eu.europa.esig.dss.asic.common.ASiCParameters;
import eu.europa.esig.dss.asic.common.ZipUtils;
import eu.europa.esig.dss.enumerations.ASiCContainerType;
import eu.europa.esig.dss.exception.IllegalInputException;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.InMemoryDocument;
import eu.europa.esig.dss.model.MimeType;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.utils.Utils;
import eu.europa.esig.dss.validation.ManifestEntry;
import eu.europa.esig.dss.validation.ManifestFile;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ASiCUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ASiCUtils.class);
    public static final String MANIFEST_FILENAME = "Manifest";
    public static final String ASIC_MANIFEST_FILENAME = "ASiCManifest";
    public static final String ASIC_ARCHIVE_MANIFEST_FILENAME = "ASiCArchiveManifest";
    public static final String MIME_TYPE = "mimetype";
    public static final String MIME_TYPE_COMMENT = "mimetype=";
    public static final String META_INF_FOLDER = "META-INF/";
    public static final String PACKAGE_ZIP = "package.zip";
    public static final String SIGNATURE_FILENAME = "signature";
    public static final String TIMESTAMP_FILENAME = "timestamp";
    public static final String CADES_SIGNATURE_EXTENSION = ".p7s";
    public static final String TST_EXTENSION = ".tst";
    public static final String XML_EXTENSION = ".xml";

    private ASiCUtils() {
    }

    public static boolean isSignature(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(SIGNATURE_FILENAME) && !entryName.contains(MANIFEST_FILENAME);
    }

    public static boolean isTimestamp(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(TIMESTAMP_FILENAME) && entryName.endsWith(TST_EXTENSION);
    }

    public static String getMimeTypeString(ASiCParameters asicParameters) {
        MimeType mimeType = ASiCUtils.getMimeType(asicParameters);
        return mimeType.getMimeTypeString();
    }

    public static String getZipComment(ASiCParameters asicParameters) {
        if (asicParameters.isZipComment()) {
            return ASiCUtils.getZipComment(ASiCUtils.getMimeTypeString(asicParameters));
        }
        return "";
    }

    public static String getZipComment(MimeType mimeType) {
        return ASiCUtils.getZipComment(mimeType.getMimeTypeString());
    }

    public static String getZipComment(String mimeTypeString) {
        return MIME_TYPE_COMMENT + mimeTypeString;
    }

    public static boolean isASiCMimeType(MimeType mimeType) {
        return MimeType.ASICS.equals(mimeType) || MimeType.ASICE.equals(mimeType);
    }

    public static boolean isOpenDocumentMimeType(MimeType mimeType) {
        return MimeType.ODT.equals(mimeType) || MimeType.ODS.equals(mimeType) || MimeType.ODG.equals(mimeType) || MimeType.ODP.equals(mimeType);
    }

    public static ASiCContainerType getASiCContainerType(MimeType asicMimeType) {
        if (MimeType.ASICS.equals(asicMimeType)) {
            return ASiCContainerType.ASiC_S;
        }
        if (MimeType.ASICE.equals(asicMimeType) || ASiCUtils.isOpenDocumentMimeType(asicMimeType)) {
            return ASiCContainerType.ASiC_E;
        }
        throw new IllegalArgumentException("Not allowed mimetype '" + asicMimeType.getMimeTypeString() + "'");
    }

    public static boolean isASiCE(ASiCParameters asicParameters) {
        Objects.requireNonNull(asicParameters.getContainerType(), "ASiCContainerType must be defined!");
        return ASiCContainerType.ASiC_E.equals((Object)asicParameters.getContainerType());
    }

    public static boolean isASiCS(ASiCParameters asicParameters) {
        Objects.requireNonNull(asicParameters.getContainerType(), "ASiCContainerType must be defined!");
        return ASiCContainerType.ASiC_S.equals((Object)asicParameters.getContainerType());
    }

    public static MimeType getMimeType(ASiCParameters asicParameters) {
        if (Utils.isStringNotBlank(asicParameters.getMimeType())) {
            return MimeType.fromMimeTypeString(asicParameters.getMimeType());
        }
        return ASiCUtils.isASiCE(asicParameters) ? MimeType.ASICE : MimeType.ASICS;
    }

    public static boolean areFilesContainCorrectSignatureFileWithExtension(List<String> filenames, String extension) {
        for (String filename : filenames) {
            if (!ASiCUtils.isSignature(filename) || !filename.endsWith(extension)) continue;
            return true;
        }
        return false;
    }

    public static boolean filesContainSignatures(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isSignature(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean isAsicFileContent(List<String> filenames) {
        return ASiCUtils.areFilesContainCorrectSignatureFileWithExtension(filenames, CADES_SIGNATURE_EXTENSION) || ASiCUtils.areFilesContainCorrectSignatureFileWithExtension(filenames, XML_EXTENSION) || ASiCUtils.areFilesContainTimestamp(filenames);
    }

    public static boolean areFilesContainTimestamp(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isTimestamp(filename)) continue;
            return true;
        }
        return false;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean isZip(DSSDocument document) {
        if (document == null) {
            return false;
        }
        byte[] preamble = new byte[2];
        try (InputStream is = document.openStream();){
            int r = is.read(preamble, 0, 2);
            if (r != 2) {
                boolean bl = false;
                return bl;
            }
        }
        catch (IOException e) {
            throw new IllegalInputException("Unable to read the 2 first bytes", e);
        }
        if (preamble[0] != 80) return false;
        if (preamble[1] != 75) return false;
        return true;
    }

    public static boolean isASiCWithXAdES(List<String> filenames) {
        return ASiCUtils.areFilesContainCorrectSignatureFileWithExtension(filenames, XML_EXTENSION);
    }

    public static boolean isASiCWithCAdES(List<String> filenames) {
        return ASiCUtils.areFilesContainCorrectSignatureFileWithExtension(filenames, CADES_SIGNATURE_EXTENSION) || ASiCUtils.areFilesContainTimestamp(filenames);
    }

    public static boolean isXAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(XML_EXTENSION);
    }

    public static boolean isCAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(CADES_SIGNATURE_EXTENSION);
    }

    public static boolean isContainerOpenDocument(DSSDocument archiveContainer) {
        DSSDocument mimetype = ASiCUtils.getMimetypeDocument(archiveContainer);
        return mimetype != null && ASiCUtils.isOpenDocument(mimetype);
    }

    private static DSSDocument getMimetypeDocument(DSSDocument dssDocument) {
        List<DSSDocument> documents = ZipUtils.getInstance().extractContainerContent(dssDocument);
        for (DSSDocument document : documents) {
            if (!ASiCUtils.isMimetype(document.getName())) continue;
            return document;
        }
        return null;
    }

    public static boolean isOpenDocument(DSSDocument mimeTypeDoc) {
        MimeType mimeType = ASiCUtils.getMimeType(mimeTypeDoc);
        if (mimeTypeDoc != null) {
            return ASiCUtils.isOpenDocumentMimeType(mimeType);
        }
        return false;
    }

    public static boolean areFilesContainMimetype(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isMimetype(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean isMimetype(String entryName) {
        return MIME_TYPE.equals(entryName);
    }

    public static MimeType getMimeType(DSSDocument mimeTypeDocument) {
        MimeType mimeType;
        block9: {
            if (mimeTypeDocument == null) {
                return null;
            }
            InputStream is = mimeTypeDocument.openStream();
            try {
                byte[] byteArray = Utils.toByteArray(is);
                String mimeTypeString = new String(byteArray, StandardCharsets.UTF_8);
                mimeType = MimeType.fromMimeTypeString(mimeTypeString);
                if (is == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new IllegalInputException(String.format("Unable to read mimetype file. Reason : %s", e.getMessage()), e);
                }
            }
            is.close();
        }
        return mimeType;
    }

    public static ASiCContainerType getContainerType(DSSDocument archive, DSSDocument mimetype, String zipComment, List<DSSDocument> signedDocuments) {
        ASiCContainerType containerType = ASiCUtils.getContainerTypeFromMimeType(archive.getMimeType());
        if (containerType == null && (containerType = ASiCUtils.getContainerTypeFromMimeTypeDocument(mimetype)) == null) {
            containerType = ASiCUtils.getContainerTypeFromZipComment(zipComment);
        }
        if (containerType == null) {
            LOG.warn("Unable to define the ASiC Container type with its properties");
            if (Utils.collectionSize(signedDocuments) == 1) {
                containerType = ASiCContainerType.ASiC_S;
            } else if (Utils.collectionSize(signedDocuments) > 1) {
                containerType = ASiCContainerType.ASiC_E;
            } else {
                throw new IllegalInputException("The provided file does not contain signed documents. The signature validation is not possible");
            }
        }
        return containerType;
    }

    private static ASiCContainerType getContainerTypeFromZipComment(String zipComment) {
        int indexOf;
        if (Utils.isStringNotBlank(zipComment) && (indexOf = zipComment.indexOf(MIME_TYPE_COMMENT)) > -1) {
            String asicCommentMimeTypeString = zipComment.substring(MIME_TYPE_COMMENT.length() + indexOf);
            MimeType mimeTypeFromZipComment = MimeType.fromMimeTypeString(asicCommentMimeTypeString);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromZipComment);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeTypeDocument(DSSDocument mimetype) {
        if (mimetype != null) {
            MimeType mimeTypeFromEmbeddedFile = ASiCUtils.getMimeType(mimetype);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromEmbeddedFile);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeType(MimeType mimeType) {
        if (ASiCUtils.isASiCMimeType(mimeType)) {
            return ASiCUtils.getASiCContainerType(mimeType);
        }
        return null;
    }

    public static String getPadNumber(int num) {
        String numStr = String.valueOf(num);
        String zeroPad = "000";
        return zeroPad.substring(numStr.length()) + numStr;
    }

    public static boolean isManifest(String fileName) {
        return fileName.startsWith(META_INF_FOLDER) && fileName.contains(ASIC_MANIFEST_FILENAME) && fileName.endsWith(XML_EXTENSION);
    }

    public static boolean isArchiveManifest(String fileName) {
        return fileName.startsWith(META_INF_FOLDER) && fileName.contains(ASIC_ARCHIVE_MANIFEST_FILENAME) && fileName.endsWith(XML_EXTENSION);
    }

    public static String getNextASiCManifestName(String expectedManifestName, List<DSSDocument> existingManifests) {
        List<String> manifestNames = DSSUtils.getDocumentNames(existingManifests);
        String manifestName = null;
        for (int i = 0; i < existingManifests.size() + 1; ++i) {
            String suffix = i == 0 ? "" : String.valueOf(i);
            manifestName = META_INF_FOLDER + expectedManifestName + suffix + XML_EXTENSION;
            if (ASiCUtils.isValidName(manifestName, manifestNames)) break;
        }
        return manifestName;
    }

    private static boolean isValidName(String name, List<String> notValidNames) {
        return !notValidNames.contains(name);
    }

    public static boolean isASiCSArchive(DSSDocument document) {
        return Utils.areStringsEqual(PACKAGE_ZIP, document.getName());
    }

    public static boolean coversSignature(ManifestFile manifestFile) {
        for (ManifestEntry manifestEntry : manifestFile.getEntries()) {
            if (!ASiCUtils.isSignature(manifestEntry.getFileName())) continue;
            return true;
        }
        return false;
    }

    public static void addOrReplaceDocument(List<DSSDocument> documentList, DSSDocument newDocument) {
        for (int i = 0; i < documentList.size(); ++i) {
            if (!newDocument.getName().equals(documentList.get(i).getName())) continue;
            documentList.set(i, newDocument);
            return;
        }
        documentList.add(newDocument);
    }

    public static ASiCContent ensureMimeTypeAndZipComment(ASiCContent asicContent, ASiCParameters asicParameters) {
        MimeType mimeType = ASiCUtils.getMimeType(asicContent, asicParameters);
        if (asicContent.getMimeTypeDocument() == null) {
            DSSDocument mimetypeDocument = ASiCUtils.createMimetypeDocument(mimeType);
            asicContent.setMimeTypeDocument(mimetypeDocument);
        }
        String zipComment = ASiCUtils.getZipComment(asicContent, asicParameters, mimeType);
        if (Utils.isStringEmpty(asicContent.getZipComment())) {
            asicContent.setZipComment(zipComment);
        }
        return asicContent;
    }

    private static MimeType getMimeType(ASiCContent asicContent, ASiCParameters asicParameters) {
        MimeType mimeType = null;
        DSSDocument mimeTypeDocument = asicContent.getMimeTypeDocument();
        if (mimeTypeDocument != null) {
            mimeType = ASiCUtils.getMimeType(mimeTypeDocument);
        }
        if (mimeType == null) {
            Objects.requireNonNull(asicParameters, "ASiCParameters shall be present for the requested operation!");
            mimeType = ASiCUtils.getMimeType(asicParameters);
        }
        return mimeType;
    }

    private static String getZipComment(ASiCContent asicContent, ASiCParameters asicParameters, MimeType mimeType) {
        String zipComment = asicContent.getZipComment();
        if (Utils.isStringNotEmpty(zipComment)) {
            return zipComment;
        }
        if (asicParameters != null && asicParameters.isZipComment()) {
            return ASiCUtils.getZipComment(mimeType);
        }
        return "";
    }

    private static DSSDocument createMimetypeDocument(MimeType mimeType) {
        byte[] mimeTypeBytes = mimeType.getMimeTypeString().getBytes(StandardCharsets.UTF_8);
        return new InMemoryDocument(mimeTypeBytes, MIME_TYPE);
    }

    public static List<DSSDocument> getRootLevelSignedDocuments(ASiCContent asicContent) {
        List<DSSDocument> signedDocuments = asicContent.getSignedDocuments();
        if (Utils.isCollectionEmpty(signedDocuments)) {
            return Collections.emptyList();
        }
        if (Utils.collectionSize(signedDocuments) == 1) {
            return signedDocuments;
        }
        ArrayList<DSSDocument> rootDocuments = new ArrayList<DSSDocument>();
        for (DSSDocument document : signedDocuments) {
            if (document.getName() == null || document.getName().contains("/")) continue;
            rootDocuments.add(document);
        }
        return rootDocuments;
    }
}

