/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.pdf;

import eu.europa.esig.dss.pades.validation.PdfModification;
import eu.europa.esig.dss.pdf.AnnotationBox;
import eu.europa.esig.dss.pdf.CommonPdfModification;
import eu.europa.esig.dss.pdf.ObjectModification;
import eu.europa.esig.dss.pdf.PdfAnnotation;
import eu.europa.esig.dss.pdf.PdfArray;
import eu.europa.esig.dss.pdf.PdfDict;
import eu.europa.esig.dss.pdf.PdfDocumentReader;
import eu.europa.esig.dss.pdf.PdfObjectTree;
import eu.europa.esig.dss.pdf.visible.ImageUtils;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.utils.Utils;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfModificationDetectionUtils {
    private static final Logger LOG = LoggerFactory.getLogger(PdfModificationDetectionUtils.class);

    private PdfModificationDetectionUtils() {
    }

    public static List<PdfModification> getAnnotationOverlaps(PdfDocumentReader reader) throws IOException {
        ArrayList<PdfModification> annotationOverlaps = new ArrayList<PdfModification>();
        block0: for (int pageNumber = 1; pageNumber <= reader.getNumberOfPages(); ++pageNumber) {
            List<PdfAnnotation> pdfAnnotations = reader.getPdfAnnotations(pageNumber);
            Iterator<PdfAnnotation> iterator = pdfAnnotations.iterator();
            while (iterator.hasNext()) {
                PdfAnnotation annotation = iterator.next();
                iterator.remove();
                if (!PdfModificationDetectionUtils.isAnnotationBoxOverlapping(annotation.getAnnotationBox(), pdfAnnotations)) continue;
                annotationOverlaps.add(new CommonPdfModification(pageNumber));
                continue block0;
            }
        }
        return annotationOverlaps;
    }

    public static boolean isAnnotationBoxOverlapping(AnnotationBox annotationBox, List<PdfAnnotation> pdfAnnotations) {
        if (annotationBox.getWidth() == 0.0f || annotationBox.getHeight() == 0.0f) {
            return false;
        }
        for (PdfAnnotation pdfAnnotation : pdfAnnotations) {
            if (!annotationBox.isOverlap(pdfAnnotation)) continue;
            return true;
        }
        return false;
    }

    public static List<PdfModification> getVisualDifferences(PdfDocumentReader signedRevisionReader, PdfDocumentReader finalRevisionReader) throws IOException {
        ArrayList<PdfModification> visualDifferences = new ArrayList<PdfModification>();
        for (int pageNumber = 1; pageNumber <= signedRevisionReader.getNumberOfPages() && pageNumber <= finalRevisionReader.getNumberOfPages(); ++pageNumber) {
            List<PdfAnnotation> finalAnnotations;
            List<PdfAnnotation> signedAnnotations;
            List<PdfAnnotation> addedAnnotations;
            BufferedImage finalScreenshot;
            BufferedImage signedScreenshot = signedRevisionReader.generateImageScreenshot(pageNumber);
            if (ImageUtils.imagesEqual(signedScreenshot, finalScreenshot = finalRevisionReader.generateImageScreenshotWithoutAnnotations(pageNumber, addedAnnotations = PdfModificationDetectionUtils.getUpdatedAnnotations(signedAnnotations = signedRevisionReader.getPdfAnnotations(pageNumber), finalAnnotations = finalRevisionReader.getPdfAnnotations(pageNumber))))) continue;
            LOG.warn("A visual difference found on page {} between a signed revision and the final document!", (Object)pageNumber);
            visualDifferences.add(new CommonPdfModification(pageNumber));
        }
        return visualDifferences;
    }

    public static List<PdfModification> getPagesDifferences(PdfDocumentReader signedRevisionReader, PdfDocumentReader finalRevisionReader) {
        int signedPages = signedRevisionReader.getNumberOfPages();
        int finalPages = finalRevisionReader.getNumberOfPages();
        int maxNumberOfPages = Math.max(signedPages, finalPages);
        int minNumberOfPages = Math.min(signedPages, finalPages);
        ArrayList<PdfModification> missingPages = new ArrayList<PdfModification>();
        for (int ii = maxNumberOfPages; ii > minNumberOfPages; --ii) {
            missingPages.add(new CommonPdfModification(ii));
        }
        if (Utils.isCollectionNotEmpty(missingPages)) {
            LOG.warn("The provided PDF file contains {} additional pages against the signed revision!", (Object)(maxNumberOfPages - minNumberOfPages));
        }
        return missingPages;
    }

    private static List<PdfAnnotation> getUpdatedAnnotations(List<PdfAnnotation> signedAnnotations, List<PdfAnnotation> finalAnnotations) {
        ArrayList<PdfAnnotation> updatesAnnotations = new ArrayList<PdfAnnotation>();
        for (PdfAnnotation annotationBox : finalAnnotations) {
            if (signedAnnotations.contains(annotationBox)) continue;
            updatesAnnotations.add(annotationBox);
        }
        return updatesAnnotations;
    }

    public static Set<ObjectModification> getModificationSet(PdfDocumentReader signedRevisionReader, PdfDocumentReader finalRevisionReader) {
        LinkedHashSet<ObjectModification> modifications = new LinkedHashSet<ObjectModification>();
        PdfDict signedCatalogDict = signedRevisionReader.getCatalogDictionary();
        PdfDict finalCatalogDict = finalRevisionReader.getCatalogDictionary();
        PdfModificationDetectionUtils.compareDictsRecursively(modifications, new HashSet<String>(), new PdfObjectTree("Catalog"), signedCatalogDict, finalCatalogDict);
        return modifications;
    }

    private static void compareDictsRecursively(Set<ObjectModification> modifications, Set<String> processedObjects, PdfObjectTree objectTree, PdfDict signedDict, PdfDict finalDict) {
        String[] signedRevKeys = signedDict.list();
        String[] finalRevKeys = finalDict.list();
        for (String key : signedRevKeys) {
            Long objectNumber;
            PdfObjectTree currentObjectTree = objectTree.copy();
            if (PdfModificationDetectionUtils.isProcessedReference(processedObjects, currentObjectTree, key, objectNumber = signedDict.getObjectNumber(key))) continue;
            currentObjectTree.addKey(key);
            PdfModificationDetectionUtils.addProcessedReference(processedObjects, currentObjectTree, key, objectNumber);
            PdfModificationDetectionUtils.compareObjectsRecursively(modifications, processedObjects, currentObjectTree, key, signedDict.getObject(key), finalDict.getObject(key));
        }
        List<String> signedRevKeyList = Arrays.asList(signedRevKeys);
        for (String key : finalRevKeys) {
            PdfObjectTree currentObjectTree = objectTree.copy();
            if (signedRevKeyList.contains(key)) continue;
            currentObjectTree.addKey(key);
            Object finalObject = finalDict.getObject(key);
            if (finalObject instanceof PdfDict || finalObject instanceof PdfArray) {
                Long objectNumber = finalDict.getObjectNumber(key);
                PdfModificationDetectionUtils.addProcessedReference(processedObjects, currentObjectTree, key, objectNumber);
                modifications.add(ObjectModification.create(currentObjectTree, finalDict.getObject(key)));
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Added entry with key '{}'.", (Object)currentObjectTree);
                continue;
            }
            modifications.add(ObjectModification.modify(currentObjectTree, null, finalObject));
            if (!LOG.isDebugEnabled()) continue;
            LOG.debug("Added parameter with key name '{}'.", (Object)objectTree);
        }
        PdfModificationDetectionUtils.compareDictStreams(modifications, objectTree, signedDict, finalDict);
    }

    private static void compareObjectsRecursively(Set<ObjectModification> modifications, Set<String> processedObjects, PdfObjectTree objectTree, String key, Object signedObject, Object finalObject) {
        if (signedObject == null && finalObject != null) {
            if (finalObject instanceof PdfDict || finalObject instanceof PdfArray) {
                modifications.add(ObjectModification.create(objectTree, finalObject));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Added entry with key '{}'.", (Object)objectTree);
                }
            } else {
                modifications.add(ObjectModification.modify(objectTree, null, finalObject));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Added parameter with key name '{}'.", (Object)objectTree);
                }
            }
        } else if (signedObject != null && finalObject == null) {
            if (signedObject instanceof PdfDict || signedObject instanceof PdfArray) {
                modifications.add(ObjectModification.delete(objectTree, signedObject));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Deleted entry with key '{}'.", (Object)objectTree);
                }
            } else {
                modifications.add(ObjectModification.modify(objectTree, signedObject, null));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Deleted parameter with key name '{}'.", (Object)objectTree);
                }
            }
        } else if (signedObject != null && finalObject != null) {
            if (signedObject instanceof PdfDict && finalObject instanceof PdfDict) {
                PdfModificationDetectionUtils.compareDictsRecursively(modifications, processedObjects, objectTree, (PdfDict)signedObject, (PdfDict)finalObject);
            } else if (signedObject instanceof PdfArray && finalObject instanceof PdfArray) {
                PdfArray signedArray = (PdfArray)signedObject;
                PdfArray finalArray = (PdfArray)finalObject;
                PdfModificationDetectionUtils.compareArraysRecursively(modifications, processedObjects, objectTree, key, signedArray, finalArray, true);
                PdfModificationDetectionUtils.compareArraysRecursively(modifications, processedObjects, objectTree, key, finalArray, signedArray, false);
            } else if (signedObject instanceof String && finalObject instanceof String) {
                if (!signedObject.equals(finalObject)) {
                    modifications.add(ObjectModification.modify(objectTree, signedObject, finalObject));
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Object changed with key '{}'.", (Object)objectTree);
                    }
                }
            } else if (signedObject instanceof Number && finalObject instanceof Number) {
                if (!signedObject.equals(finalObject)) {
                    modifications.add(ObjectModification.modify(objectTree, signedObject, finalObject));
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Object changed with key '{}'.", (Object)objectTree);
                    }
                }
            } else if (signedObject instanceof Boolean && finalObject instanceof Boolean) {
                if (!signedObject.equals(finalObject)) {
                    modifications.add(ObjectModification.modify(objectTree, signedObject, finalObject));
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Object changed with key '{}'.", (Object)objectTree);
                    }
                }
            } else {
                modifications.add(ObjectModification.modify(objectTree, signedObject, finalObject));
                LOG.warn("Unsupported objects found with key '{}' of types '{}' and '{}'", objectTree, signedObject.getClass(), finalObject.getClass());
            }
        }
    }

    private static void compareArraysRecursively(Set<ObjectModification> modifications, Set<String> processedObjects, PdfObjectTree objectTree, String key, PdfArray firstArray, PdfArray secondArray, boolean signedFirst) {
        for (int i = 0; i < firstArray.size(); ++i) {
            PdfObjectTree currentObjectTree = objectTree.copy();
            Object signedRevObject = firstArray.getObject(i);
            Object finalRevObject = null;
            Long objectNumber = firstArray.getObjectNumber(i);
            if (objectNumber != null) {
                for (int j = 0; j < secondArray.size(); ++j) {
                    Long finalObjectNumber = secondArray.getObjectNumber(j);
                    if (!objectNumber.equals(finalObjectNumber)) continue;
                    finalRevObject = secondArray.getObject(j);
                }
            } else {
                finalRevObject = secondArray.getObject(i);
            }
            if (PdfModificationDetectionUtils.isProcessedReference(processedObjects, currentObjectTree, key, objectNumber)) continue;
            PdfModificationDetectionUtils.addProcessedReference(processedObjects, currentObjectTree, key, objectNumber);
            PdfModificationDetectionUtils.compareObjectsRecursively(modifications, processedObjects, currentObjectTree, key, signedFirst ? signedRevObject : finalRevObject, signedFirst ? finalRevObject : signedRevObject);
        }
    }

    private static boolean isProcessedReference(Set<String> processedObjects, PdfObjectTree objectTree, String key, Number objectNumber) {
        return processedObjects.contains(key + objectNumber) || objectTree.isProcessedReference(objectNumber);
    }

    private static void addProcessedReference(Set<String> processedObjects, PdfObjectTree objectTree, String key, Number objectNumber) {
        if (objectNumber != null) {
            processedObjects.add(key + objectNumber);
            objectTree.addReference(objectNumber);
        }
    }

    private static void compareDictStreams(Set<ObjectModification> modifications, PdfObjectTree objectTree, PdfDict signedDict, PdfDict finalDict) {
        PdfObjectTree currentObjectTree = objectTree.copy();
        currentObjectTree.setStream();
        byte[] signedStream = PdfModificationDetectionUtils.getStreamBytesSecurely(signedDict);
        byte[] finalBytes = PdfModificationDetectionUtils.getStreamBytesSecurely(finalDict);
        if (Utils.isArrayEmpty(signedStream) && Utils.isArrayNotEmpty(finalBytes)) {
            modifications.add(ObjectModification.create(currentObjectTree, finalDict));
            if (LOG.isDebugEnabled()) {
                LOG.debug("A stream has been added '{}'.", (Object)currentObjectTree);
            }
        } else if (Utils.isArrayNotEmpty(signedStream) && Utils.isArrayEmpty(finalBytes)) {
            modifications.add(ObjectModification.delete(currentObjectTree, signedDict));
            if (LOG.isDebugEnabled()) {
                LOG.debug("A stream has been removed '{}'.", (Object)currentObjectTree);
            }
        } else if (Utils.isArrayNotEmpty(signedStream) && Utils.isArrayNotEmpty(finalBytes) && !Arrays.equals(signedStream, finalBytes)) {
            modifications.add(ObjectModification.modify(currentObjectTree, signedDict, finalDict));
            if (LOG.isDebugEnabled()) {
                LOG.debug("A stream has been modified '{}'.", (Object)currentObjectTree);
            }
        }
    }

    private static byte[] getStreamBytesSecurely(PdfDict pdfDict) {
        try {
            return pdfDict.getStreamBytes();
        }
        catch (IOException e) {
            LOG.debug("Unable to compare underlying stream binaries. Reason : {}", (Object)e.getMessage());
            return DSSUtils.EMPTY_BYTE_ARRAY;
        }
    }
}

